(in-package "CL-USER")
;; (load "dbmc-structs")
;; (load "helper-functions")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  special constants section                ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *primitives* '(=
		       <
		       >
		       <=
		       >=
		       <-> 
		       -> 
		       and 
		       or 
		       xor 
		       not
		       add
		       sub
		       neg
		       + 
		       - 
		       inc 
		       dec
		       mod+
		       mod-
		       *
		       mult
		       mod*
		       const 
		       if 
		       cond 
		       >> 
		       <<
		       >>>
		       <<<
		       cat 
		       type 
		       bit 
		       bits
		       get
		       set
		       ext 
		       local 
		       foldl 
		       foldr 
		       AF 
		       AG 
		       next
		       mv
		       mv-let))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; helper functions                          ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; changed by roma on 9 March - no variable can start
; with 0b and have 0 and 1 
; changed on 18 May to incorporate Verilog representation

;checks if n is a valid name in desc d. basically, it checks to make
;sure n hasn't been taken already.
(defun valid-namep (n d)
  (not (or (assoc n (desc-consts d))
           (assoc n (desc-defs d))
	   (assoc n (desc-functs d))
	   (member n *primitives*)
	   (char= (elt (string n) 0) #\_)
	   (constp n))))
;; 	   (and (>= (char-code (elt (string-downcase n) 0)) 48) 
;; 		(<= (char-code (elt (string-downcase n) 0)) 57) 
;; 		(char-equal (elt (string-downcase n) 1) #\h)
;; 		(dotimes (x (- (length (string n)) 2) nil)
;; 		  (unless (or (and (char>= (elt (string-downcase n) (+ 2 x)) #\0)
;; 				   (char<= (elt (string-downcase n) (+ 2 x)) #\9))
;; 			      (and (char>= (elt (string-downcase n) (+ 2 x)) #\a)
;; 				   (char<= (elt (string-downcase n) (+ 2 x)) #\f)))
;; 		    (return t))))
;; 	   (and (>= (char-code (elt (string-downcase n) 0)) 48)
;; 		(<= (char-code (elt (string-downcase n) 0)) 57) 		
;; 		(char-equal (elt (string-downcase n) 1) #\o)
;; 		(dotimes (x (- (length (string n)) 2) nil)
;; 		  (unless (and (char>= (elt (string-downcase n) (+ 2 x)) #\0)
;; 			       (char<= (elt (string-downcase n) (+ 2 x)) #\7))
;; 		    (return t))))
;; 	   (and (char-equal (elt (string n) (- (length (string n)) 1)) #\u)
;; 		(dotimes (x (- (length (string n)) 1) nil)
;; 		  (unless (or (char>= (elt (string n) x) #\0)
;; 			      (char<= (elt (string n) x) #\9))
;; 		    (return t))))
;;            (and (>= (char-code (elt (string-downcase n) 0)) 48)
;; 		(<= (char-code (elt (string-downcase n) 0)) 57) 		
;; 		(char-equal (elt (string n) 1) #\b)
;; 		(dotimes (x (- (length (string n)) 2) nil)
;; 		  (unless (or (char-equal (elt (string n) (+ 2 x)) #\0)
;; 			      (char-equal (elt (string n) (+ 2 x)) #\1))
;; 		    (return t)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; consts section                            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;gets the value of a constant named c in desc d.
(defun get-const-val (c d)
  (second (assoc c (desc-consts d))))

; changed by roma on 8 March - added binary nos to constants - can start
; with 0b and have 0 and 1 

(defun constp (form)
  (if (atom form)
      (or (integerp form)
	  (or (when (char-equal (elt (string form) (- (length (string form)) 1)) #\u)
		(let ((len (length (string form))))
		  (and (> len 1)
		       (dotimes (x (- len 1) t)
			 (unless (and (char>= (elt (string form) x) #\0)
				      (char<= (elt (string form) x) #\9))
			   (return nil))))))
	      (and (>= (char-code (elt (string-downcase form) 0)) 48)
		   (<= (char-code (elt (string-downcase form) 0)) 57)
		   (char-equal (elt (string form) 1) #\b)
		   (dotimes (x (- (length (string form)) 2) t)
		     (unless (or (char= (elt (string form) (+ 2 x)) #\0)
				 (char= (elt (string form) (+ 2 x)) #\1))
		       (return nil))))
	      (and (>= (char-code (elt (string-downcase form) 0)) 48)
		   (<= (char-code (elt (string-downcase form) 0)) 57)
		   (char-equal (elt (string-downcase form) 1) #\o)
		   (dotimes (x (- (length (string form)) 2) t)
		     (unless (or (and (char>= (elt (string-downcase form) (+ 2 x)) #\0)
				      (char<= (elt (string-downcase form) (+ 2 x)) #\7)))
		       (return nil))))
	      (numberp form)
	      (and (>= (char-code (elt (string-downcase form) 0)) 48)
		   (<= (char-code (elt (string-downcase form) 0)) 57) 		   
		   (char-equal (elt (string-downcase form) 1) #\x)
		   (dotimes (x (- (length (string form)) 2) t)
		     (unless (or (and (char>= (elt (string-downcase form) (+ 2 x)) #\0)
				      (char<= (elt (string-downcase form) (+ 2 x)) #\9))
				 (and (char>= (elt (string-downcase form) (+ 2 x)) #\a)
				      (char<= (elt (string-downcase form) (+ 2 x)) #\f)))
		       (return nil))))))	
    (and (equal (car form) 'const)
	 (dolist (x (cdr form) t)
	   (unless (or (0p x)
		       (1p x))
	     (return nil))))))

;checks if c is a valid constant definition in desc d.
(defun const-defp (c d)
  (if (and (listp c)
	   (equal (length c) 2)
	   (symbolp (first c))
	   (valid-namep (first c) d)
	   (constp (second c)))
      (setf (desc-consts d)
	    (cons c
		  (desc-consts d)))
    (format t "~& Incorrect constant definition: ~a" c)))

;checks that form is a list of valid constant definitions.
(defun const-listp (form d)
  (if (endp form)
      t
    (and (const-defp (car form) d)
	 (const-listp (cdr form) d))))

;checks that form is a valid consts section in d.
(defun const-sectionp (form d)
  (and (equal (car form) :consts)
       (consp (cdr form))
       (const-listp (cdr form) d)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; replacing constants                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;replaces form with its constant value if it is a constant.
(defun constant-val (form d)
  (let ((c (get-const-val form d)))
    (if c
	c
      form)))

;replaces all the constants in form with their respective values.
(defun replace-constants (form d)
  (if (atom form)
      (constant-val form d)
    (cons (replace-constants (car form) d)
	  (replace-constants (cdr form) d))))

;replaces all the constants at the "top level". That is, it replaces
;form if it is a constant. If form is a function application, it looks
;at each argument and replaces the ones that are constants. Unlike
;replace-constants, though, this function will not dive into the the
;function arguments to find more constants.
(defun tl-replace-constants (form d)
  (cond ((atom form) 
	 (constant-val form d))
	((atom (car form))
	 (cons (constant-val (car form) d)
	       (tl-replace-constants (cdr form) d)))
	(t
	 (cons (car form)
	       (tl-replace-constants (cdr form) d)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vars section                              ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;tells whether varname is a variable in the environment e.
(declaim (ftype (function (symbol list) boolean) isvar))
(defun isvar (varname e)
  (when (assoc varname e) t))

;returns the name from a variable definition, v.
(defun var-name (v)
  (first v))

;returns the type from a variable definition, v.
(defun var-bits (v)
  (second v))

;returns the word-size from a variable definition, v.
(defun var-wordsize (v)
  (third v))

;tells whether n is a valid varname given a list of already defined
;vars and a description, d.
(defun valid-varnamep (n vars d)
  (and (symbolp n)
       (valid-namep n d)
       (not (member n vars))))

;checks for a valid variable definition
; changed by roma on Nov 17,05 to include word-size
(defun var-defp (v vars d)
;  (format t "~& v: ~a " v)
  (cond ((consp v)
	 (if (and (listp v)
		  (valid-varnamep (var-name v) vars d)
		  (posp (constant-val (var-bits v) d))
		  (or (equal (length v) 2)
		      (equal (length v) 3)))
	     (if (equal (length v) 3)
		 (if (posp (constant-val (var-wordsize v) d))
		     t
		   (format t "~& Word size should be constant, you wrote: ~a" 
			   (var-wordsize v)))
	       t)
	   (format t "~& Incorrect variable definition: ~a" v)))
	(t 
	 (if (valid-varnamep v vars d)
	     t
	   (format t "~& Invalid variable name: ~a" v)))))

;replaces constants in a variable definition and changes definitions
;without explicit type into definitions with explicit type 1.
(defun fix-var (v d)
  (let ((v (replace-constants v d)))
    (cond ((atom v) (cons v '(bv 1)))
	  ((endp (cdr v)) (cons (car v) '(bv 1)))
	  ((endp (cddr v)) (cons (first v) `(bv ,(second v))))
	  (t (cons (first v) `(mem ,(second v) ,(third v)))))))

;calls fix-var on a list of variable definitions.
(defun fix-var-list (vlist d)
  (let ((f #'(lambda (x) (fix-var x d))))
    (mapcar f vlist)))

;checks for a list of valid variable definitions
(defun var-listp (form vars d)
  (dolist (v form t)
    (if (var-defp v vars d)
	(setf vars (cons (var-name (fix-var (car form) d)) vars))
      (return nil))))

;checks for a valid var section
(defun var-sectionp (form d)
  ;(format t "~&var-sectionp: e=~a" e)
  (cond ((not (equal (car form) :vars))
	 (format t "~&Improper section header: ~a" (car form)))
	((not (listp (cdr form)))
	 (format t "~&:vars section must be a listp"))
	(t (var-listp (cdr form) nil d))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; tests for formulas                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;recognizer for simple formulas: returns form's type if it is a valid simple formula
;a simple formula is a constant, variable, or bit-slice of a variable.
(defun simple-formulap (form e d)
  (let ((cform (tl-replace-constants form d)))
    (if (constp cform)
	t
      (if (atom cform) 
	  (if (isvar form e)
	      t
	    (format t "~&Unknown variable: ~a. env: ~a" form e))
	(let ((i1 (constant-val (second cform) d))
	      (i2 (constant-val (third cform) d)))
	  (cond ((not (isvar (car cform) e))
		 (format t "~&Unknown variable: ~a" (car cform)))
		((and (= (length cform) 2) 
		      (natp i1))
		 t)
		((and (= (length cform) 3)
		      (natp i1)
		      (natp i2)
		      (< i1 i2))
		 t)
		(t (format t 
			   "~&Ill-formed variable dereference: ~a" 
			   form))))))))

;makes sure all the items in form are valid formulas
(defun basic-formula-listp (predicate args e d)
  (declare (type function predicate))
  (dolist (a args t)
    (unless (funcall predicate a e d) 
      (return nil))))

;;;local;;;

;checks the optional first part of a local, which declares variables
;that will be defined in a bitwise manner in the local bindings.
(defun localp-aux-1 (a d)
  (let ((lst nil))
    (format t "~& local eg: ~a" lst)
    (dolist (x a t)
      (if (var-defp x lst d)
	  (when (not (and (consp x)
			  (> (the fixnum (second x)) 1)))
	    (return (format t 
			    "~&In the first part of a local, variables must be of size > 1. You wrote: ~a"
			    x)))
	(return nil)))))

;checks the bindings of a local.
(defun localp-aux-2 (a predicate vars e d)
  (declare (type function predicate))
  (dolist (x a e)
    ;; (format t "~&localp-aux-2: x: ~A~%" x)
    (if (and (true-listp x)
	     (or (atom (car x))
		 (true-listp (car x))))
	(let ((vdefs (car x))
	      (len (length x)))
	  (cond ((= len 3)
		 (if (and (valid-varnamep vdefs vars d)
			  (posp (second x)))
		     (if (funcall predicate (third x) e d)
			 (setf e (cons (list vdefs (second x)) e))
		       (return nil))
		   (return (format t "~&Invalid binding 1: ~a" x))))
		((= len 4)
		 (if (and (valid-varnamep vdefs vars d)
			  (posp (second x))
			  (posp (third x)))
		     (if (funcall predicate (fourth x) e d)
			 (setf e (cons (list vdefs (second x)) e))
		       (return nil))
		   (return (format t "~&Invalid binding 8: ~a" x))))
		((not (= len 2))
		 (return (format t "~&Invalid binding 2: ~a" x)))
		((atom vdefs)
		 (if (or (member vdefs vars)
			 (valid-varnamep vdefs vars d))
		     (if (funcall predicate (second x) e d)
			 (setf e (cons (list vdefs 0) e))
		       (return nil))
		   (return (format t "~&Invalid binding 3: ~a" x))))
		((and (= (length vdefs) 2)
		      (member (first vdefs) vars)
		      (natp (second vdefs)))
		 t)
		((and (= (length vdefs) 3)
		      (member (first vdefs) vars)
		      (natp (second vdefs))
		      (posp (third vdefs)))
		 t)
		(t
		 (let* ((vars2 vars)
			(ret (dolist (b vdefs t)
;;                               (format t "~&b: ~a" b)
			      (cond ((atom b)
				     (cond ((member b vars) t)
					   ((valid-varnamep b vars d)
					    (setf e (cons (list b 1) e)))
					   (t (return (format t "~&Incorrect binding 4: ~a" x)))))
				    ((and (member (car b) vars)
					  (= (length b) 2))
				     (unless (natp (second b))
				       (return (format t "~&Incorrect binding 5: ~a" x))))
				    ((and (member (car b) vars)
					  (= (length b) 3))
				     (unless (and (natp (second b))
						  (posp (third b)))
				       (return (format t "~&Incorrect binding 6: ~a" x))))
				    (t
				     (cond ((and (= (length b) 2)
						 (valid-varnamep (car b) vars2 d)
						 (posp (second b)))
					    (setf e (cons b e))
					    (setf vars2 (cons (car b) vars2)))
					   ((and (= (length b) 3)
						 (valid-varnamep (car b) vars2 d)
						 (posp (second b))
						 (posp (third b)))
					    (setf e (cons b e))
					    (setf vars2 (cons (car b) vars2)))
					   (t (return (format t "~&Incorrect binding 7: ~a" x)))))))))
		   (unless ret (return nil)))))
	  (unless (funcall predicate (car (last x)) e d)
	    (return nil)))
      (format t "~&Incorrect binding 8: ~a" x))))

;verifies that args are those of a local.
(defun localp (predicate args e d)
  (declare (type function predicate)
	   (type list args))
  (let ((len (length args)))
    (if (or (= len 2)
	    (= len 3))
	(let ((a1 (if (= len 3) (first args) nil))
	      (a2 (if (= len 3) (second args) (first args)))
	      (a3 (if (= len 3) (third args) (second args))))
	  (if (var-listp a1 (append (desc-consts d) (desc-functs d)) d)
	      (let ((e2 (localp-aux-2 a2 
				      predicate 
				      (mapcar #'car a1)
				      (append a1 e) d)))
		
		(when e2
		  (funcall predicate a3 e2 d)))
	    (format t "Incorrect local bit-vector definitions: ~a" a1)))
      (format t "locals must have 2 or 3 arguments. You wrote (local ~a)~%" args))))
		
(defun length-equal (lst x)
  (declare (type fixnum x)
	   (type list lst))
  (cond ((zp x) (endp lst))
	((endp lst) nil)
	(t (length-equal (cdr lst) (1- x)))))

;this is the meat of the formula syntax checker. it checks the stuff
;that is common to init, trans, and spec formulas.  basically, trans
;and spec specific checks are made in trans-formulap and spec-formulap
;(which check for nexts, etc.), so basic-formulap calls the
;appropriate specific formula checker, which is passed to it in the
;predicate field. form is the formula being checked, e is the variable
;environment, and d is the description.
(defun basic-formulap (predicate form e d)
  (declare (type function predicate)
	   (type list e)
	   (type desc d))
  ;(format t "~&basic-formulap form: ~a" form)
  (let ((cform form));(tl-replace-constants form d)))
    (if (atom cform)
	(simple-formulap form e d)
      (let ((funct (car cform))
	    (args (cdr cform)))
	(cond ((or (equal funct 'const)
		   (assoc funct e))
	       (simple-formulap form e d))
	      ((eq funct 'type)
	       (if (= (length args) 2)
		   (if (integerp (first args))
		       (if (posp (second args))
			   t
			 (format t 
				 "~&The second argument to type must be a positive integer. You wrote ~a."
				 form))
		     (format t
			     "~&The first argument to type must be an integer. You wrote ~a."
			     form))
		 (format t
			 "~&Type takes 2 arguments. You have ~a."
			 form)))
;; changed by roma on Nov 23,05 to include the get and set functions 
;; for memory
;; (get mem addr [words]), where mem is a variable, addr is # or bv, [words] is #
;; (set mem addr value) , value is a bit vector of size=multiple of wordsize
	      ((eq funct 'get)
	       (cond ((equal (length args) 2)
		      (when (funcall predicate (first args) e d)
			(when (funcall predicate (second args) e d)
			  t)))
		     ((equal (length args) 3)
		      (and (funcall predicate (first args) e d)
			   (funcall predicate (second args) e d)
			   (or (numberp (third args))
			       (format t "~& The third argument has to be number a. You wrote: ~a" (third args)))))
		     (t (format t "~&get takes 2 or 3 arguments. You wrote ~a." 
				 form))))
	      ((eq funct 'set)
	       (if (equal (length args) 3)
		   (and (funcall predicate (first args) e d) 
			(funcall predicate (second args) e d)
			(funcall predicate (third args) e d))
;; 		 (if (and (constp (third args))
;; 				(equal (mod (var-bits (third args)) 
;; 					    (var-wordsize (first args))) 0))
;; 			   t
;; 			 (format t "~& The third argument is not a bit-vector or its type is
;; not a multiple of word-size of ~a, you wrote: ~a" (first args) (third args)))))
		 (format t "~&set takes 3 arguments. You wrote ~a." 
			 form)))
	      ((eq funct 'ext)
	       (if (= (length args) 2)
		   (when (funcall predicate (first args) e d)
		     (if (natp (second args))
			 t
		       (format t
			       "~&The second argument to ext must be a natural number. You wrote ~a."
			       form)))
		 (format t "~&ext takes 2 arguments. You wrote ~a." form)))
	      ((member funct '(- neg))
	       (let ((len (length form)))
		 (if (or (= len 3) (= len 2))
		     (basic-formula-listp predicate args e d)
		   (format t "~&- takes 1 or 2 arguments. You wrote ~a." form))))
	      ((member funct '(-> <-> < > <= >= sub mod-))
		(if (= (length form) 3)
		   (basic-formula-listp predicate args e d)
		 (format t "~&~a takes 2 arguments. You wrote ~a." funct form)))
	      ((member funct '(>> << >>> <<<))
	       (if (= (length form) 3)
		   (if (posp (second args))
		       (funcall predicate (first args) e d)
		     (format t "~&the second argument to ~a must be a positive integer. You wrote ~a." funct form))
		 (format t "~a takes 2 arguments. You wrote ~a." funct form)))
	      ((eq funct 'if)
	       (if (= (length args) 3)
		   (basic-formula-listp predicate args e d)
		 (format t "~&if takes 3 arguments. You wrote ~a." form)))
	      ((eq funct 'cond)
	       (if (>= (length args) 1)
		   (when (dolist (f (mapcar #'car args) t)
			   (unless (funcall predicate f e d) (return nil)))
		     (basic-formula-listp predicate (mapcar #'cadr args) e d))
		 (format t "cond requires at least 1 argument. You wrote ~a" form)))
	      ((or (eq funct 'not)
		   (eq funct 'inc)
		   (eq funct 'dec))
	       (if (= (length form) 2)
		   (funcall predicate (car args) e d)
		 (format t "~&~a takes 1 argument. You wrote ~a" funct form)))
	      ((or (eq funct 'foldl)
		   (eq funct 'foldr))
	       (and (or (member (car args) '(= <-> and or xor))
			(get-funct (car args) d))
		    (funcall predicate (second args) e d)))
	      ((eq funct 'local)
	       (localp predicate args e d))
	      ((eq funct 'mv)
	       (if (or (endp args)
		       (endp (cdr args)))
		   (format t "~&mv must return at least 2 values. You wrote ~a." form)
		 (basic-formula-listp predicate args e d)))
	      ((eq funct 'mv-let)
	       (if (length-equal args 3)
		   (and (var-listp (first args) (append (desc-consts d) (desc-functs d)) d)
			(basic-formulap predicate (second args) e d)
			(basic-formulap predicate (third args) (append (fix-var-list (first args) d) e) d))
		 (format t "~&mv-let takes 3 arguments. You wrote ~A." form)))
	      ((member funct *primitives*)
	       (if (>= (length args) 2)
		   (basic-formula-listp predicate args e d)
		 (format t "~&~a takes at least 2 arguments. You wrote ~a" funct form)))
	      (t
	       (let ((f (get-funct funct d)))
		 (if f
		     (let ((len (length (funct-params f))))
		       (if (= len
			      (length args))
			   (basic-formula-listp predicate args e d)
			 (format t "~&~a takes ~a arguments. You wrote ~a" funct len form)))
		   (format t "~&Unknown function or variable name: ~a" funct)))))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; compute formula                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;check that form is an init formula
(defun compute-formulap (form e d)
  (or (and (consp form)
	   (eq (car form) '_zero_mem)
	   (or (and (= (length (cdr form)) 2)
		    (posp (first (cdr form)))
		    (posp (second (cdr form))))
	       (format t "~&_zero_mem takes 2 positive integers (number of words and wordsize). You wrote ~A~%" form)))
      (basic-formulap #'compute-formulap form e d)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; init section                              ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;check that form is an init formula
(defun init-formulap (form e d)
  (basic-formulap #'init-formulap form e d))

;check for an init section
(defun init-sectionp (form d)
  (and (listp form)
       (equal (length form) 2)
       (equal (car form) :init)
       (init-formulap (replace-constants (second form) d) 
		      (append (desc-defs d) (desc-vars d))
		      d)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; trans section                             ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;checks that form is a trans formula.
(defun trans-formulap (form e d)
  (if (and (consp form)
	   (equal (car form) 'next))
      (if (= (length form) 2)
          (let ((cform (tl-replace-constants (second form) d)))
            (if (constp cform)
                (format t "~&You cannot take the next value of a constant.")
              (if (atom cform) 
                  (cond ((isvar cform (desc-defs d)) 
                         (format t "~&You cannot find the next value of a :definitions variable. You wrote: ~a" form))
                        ((not (isvar cform e)) (format t "~&Unknown variable: ~a" cform))
                        (t t))
                (let ((i1 (constant-val (second cform) d))
                      (i2 (constant-val (third cform) d)))
                  (cond ((not (isvar (car cform) e))
                         (format t "~&Unknown variable: ~a" (car cform)))
                        ((and (= (length cform) 2) 
                              (natp i1))
                         t)
                        ((and (= (length cform) 3) 
                              (natp i1)
                              (natp i2)
                              (< i1 i2))
                         t)
                        (t (format t 
                                   "~&Ill-formed variable dereference: ~a" 
                                   form)))))))
	(format t "~&next takes 1 argument. You wrote ~a" form))
    (basic-formulap #'trans-formulap form e d)))

;checks that form is a trans section.
(defun trans-sectionp (form d)
  (cond ((not (listp form)) (format t "~&trans section must be a listp"))
        ((not (equal (length form) 2)) (format t "~&trans section must contain only one formula."))
        ((not (equal (first form) :trans)) 
         (format t "~&trans form must begin with \":trans\". You wrote ~a" (first form)))
        (t (trans-formulap (replace-constants (second form) d) 
                           (append (desc-defs d) (desc-vars d)) d))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; spec section                              ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;checks that form is a spec form.
(defun spec-formulap (form e d)
  (if (and (consp form)
	   (or (equal (car form) 'AG)
	       (equal (car form) 'AF)))
      (if (= (length form) 2)
	  (trans-formulap (second form) e  d)
	(format t "~&~a takes 1 argument. You wrote ~a" (car form) form))
    (trans-formulap form e d)))

;checks that form is a spec section.
(defun spec-sectionp (form d)
  (and (listp form)
       (equal (length form) 2)
       (equal (car form) :spec)
       (spec-formulap (replace-constants (second form) d)
                      (append (desc-defs d) (desc-vars d)) 
                      d)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; functs section                            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;adds function f to desc d.
(defun add-funct (f d)
  (setf (desc-functs d)
	(append (desc-functs d)
		(list (cons (car f)
			    (make-funct :name (car f)
					:type (fix-function-type (second f))
;; 					  :bits (if (constp (second sig))
;; 						    (* (first sig) (second sig))
;; 						  (first sig))
;; 					  :wordsize (if (constp (second sig))
;; 							(second sig)
;; 						      1)
					:params (fix-var-list (third f) d)
					:body (replace-constants (fourth f) d)))))))

(defun fix-function-type2 (lst tp)
  (if (endp lst)
      (cons 'mv (reverse tp))
    (let ((x (fix-function-type1 (car lst) nil)))
      (when x (fix-function-type2 (cdr lst) (cons x tp))))))

(defun fix-function-type1 (tp allow-mv?)
  (cond ((atom tp) nil)
	((consp (first tp))
	 (when allow-mv? 
	   (fix-function-type2 tp nil)))
	((length-equal tp 2)
	 (when (and (posp (first tp))
		    (posp (second tp)))
	   (cons 'mem tp)))
	((length-equal tp 1)
	 (when (posp (first tp))
	   (cons 'bv tp)))))

(defun fix-function-type (tp)
  (fix-function-type1 tp t))

;checks that form is a valid function definition.
; changed by roma on Nov 17,05 to include word-size
(defun function-defp (form d)
;;  (format t "~&function-defp: name:~a, form=~a" (car form) (replace-constants (second form) d))
  (if (length-equal form 4)
      (let* ((name (first form))
	     (type (fix-function-type (second form)))
	     (params (third form))
	     (body (fourth form)))
	(cond ((not (valid-namep name d))
	       (format t "~&~a is already defined" name))
	      ((not (symbolp name))
	       (format t "~&Not a valid function name: ~a" name))
	      ((not type)
	       (format t "~&Invalid function type: ~a" type))
	      ;; ((if (constp (cadr sig))
;; 		   (if (not (posp (cadr sig)))
;; 		       (format t "~& Word size should be positive, you wrote: ~a" 
;; 			       (cadr sig))
;; 		     t)
;; 		 t))
	      (t
	       (and (var-listp params nil d)
		    (init-formulap body (fix-var-list params d) d)))))
    (format t "~&Incorrect function definition format: ~a" form)))

(defun function-listp (lst d)
  (if (listp lst)
      (dolist (f lst t)
	(if (function-defp f d)
	    (add-funct f d)
	  (return nil)))
    (format t "~&Improper function list: ~A~%" lst)))

;checks that form is a valid functions section.
(defun function-sectionp (form d)
  (and (listp form)
       (equal (car form) :functions)
       (function-listp (cdr form) d)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; defs section                              ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;checks that list is a valid list of definitions.
(defun definition-listp (list d)
  (let ((defvars nil)
        (vars (desc-vars d)))
    (dolist (def list t)
      (cond ((not (listp def)) 
             (return (format t "~&Each definition must be a pair. You wrote: ~a" def)))
            ((not (valid-varnamep (first def) defvars d)) 
             (return (format t "~&~a is already defined." (first def))))
            ((not (init-formulap (second def) vars d)) (return nil))
            (t (setf vars (cons (list (first def) 0) vars))
               (setf defvars (cons (cons (first def) 0) vars)))))))

;checks that form is a valid definitions section.
(defun definition-sectionp (form d)
  (when (and (listp form)
             (equal (car form) :definitions)
             (definition-listp (cdr form) d))
    (setf (desc-defs d) (cdr form))
    t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; entire description                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;verifies that form contains valid init, trans, and spec sections.
(defun init-trans-specp (form d)
  (when (init-sectionp (first form) d)
      (setf (desc-init d) (replace-constants (second (first form)) d))
      (when (trans-sectionp (second form) d)
        (setf (desc-trans d) (replace-constants (second (second form)) d))
        (when (spec-sectionp (third form) d)
          (setf (desc-spec d) (replace-constants (second (third form)) d))
          t))))

;tests form to be sure it is a valid specification in our input language
;minus the optional functions section
(defun no-funct-specp (form d)
  (when (and (listp form)
	     (or (= (length form) 4)
                 (= (length form) 5))
	     (var-sectionp (first form) d))
    (setf (desc-vars d) (fix-var-list (cdar form) d))
    (if (equal (car (second form)) :definitions) 
        (and (definition-sectionp (second form) d)
             (init-trans-specp (cddr form) d))
      (init-trans-specp (cdr form) d))))

;checks to be sure that form is a valid description, minus a consts section.
(defun no-consts-specp (form d)
  (if (equal (caar form) :functions)
      (and (function-sectionp (car form) d)
	   (no-funct-specp (cdr form) d))
    (no-funct-specp form d)))

;syntax checker top level function.
(defun specp (form d)
  (if (equal (caar form) :consts)
      (and (const-sectionp (car form) d)
	   (no-consts-specp (cdr form) d))
    (no-consts-specp form d)))

(defun formp (vars functs form d)
  (when (function-listp functs d)
    (when (var-listp vars nil d)
      (setf (desc-vars d) (fix-var-list vars d))
      (init-formulap form (desc-vars d) d))))
